#!/bin/sh
#
# This is a shell script installer for TouchKit controller.
#
# Company: eGalax_eMPIA Technology Inc.
# Version: 1.00.0325
# Released: 2008/03/25
#

version="1.00.0325"
driver="TouchKit.tar.gz"
utility="TouchKit"
installpath="/usr/local"
shortcutpath="/usr/bin"
mousepath=`find /usr -name mouse_drv*`
xpath=${mousepath%/*} # if the path can NOT be found, how to do???
mouse=${mousepath##*/}
xfilepath="/etc/X11/xorg.conf"
xfilepath2="/etc/X11/XF86Config"
identifier="EETI"
paramfile="/var/lib/eeti.param"
usbfile="usb.info"
checkusb="lsusb -v -d 0eef:0001"
trash="/dev/null"
checkmod="lsmod"
modfile="mod.info"
blacklistpath=`find /etc -name blacklist`

ShowTitle() {
    echo ""
    echo "(*) Linux driver installer for TouchKit controller "
    echo ""
}

CheckPermission() {
    echo -n "(I) Checking user permission:"
    account=`whoami`

    if [ ${account} = "root" ]; then
        echo " ${account}, you are the supervisor."
    else
        echo " ${account}, you are NOT the supervisor."
        echo "(E) The root permission is required to run this installer.\n"
        exit 1
    fi
}

ExtractDriver() {
    if [ -e ${installpath}/${driver} ]; then
        rm -f ${installpath}/${driver}
    fi

    cp -f ${driver} ${installpath}
    ( cd ${installpath}; tar -zxf ${driver}; rm -f ${driver} )

    temppath=`find ${installpath} -name ${utility}`
    driverpath=${temppath%/*}
    echo "(I) Extract TouchKit driver package to ${driverpath}."

    if [ -d ${shortcutpath} ]; then
        echo "(I) Create ${utility} utility shortcut in ${shortcutpath}."
        ln -sf ${driverpath}/${utility} ${shortcutpath}
    else
        echo "(W) The shortcut can NOT be created in ${shortcutpath}."
    fi
}

CopyXorgMudule() {
    if [ -z ${mousepath} ]; then
        echo "(E) No X input modules directory found.\n"
        exit 1
    fi

    if [ ${mouse} = "mouse_drv.so" ]; then
        xmodule="egalax_drv.so"
    else
        xmodule="egalax_drv.o"
    fi

    if [ -e ${xpath}/${xmodule} ]; then
        rm -f ${xpath}/${xmodule}
    fi

    echo "(I) Copy X module ${xmodule} to ${xpath}."
    cp -f ${driverpath}/${xmodule} ${xpath}
}

Add2Blacklist() {
    if [ -z ${blacklistpath} ]; then
        echo "(W) No blacklist file found in /etc."
    else
        grep -q $1 ${blacklistpath}
        if [ $? -eq 1 ]; then
            filelines=`cat ${blacklistpath} | wc -l`
            echo "(I) Add kernel module $1 into ${blacklistpath}"
            sed -i ''${filelines}'a\# add by eeti\nblacklist '$1'' ${blacklistpath}
        else
            echo "(I) The kernel module $1 has been added in ${blacklistpath}."
        fi
    fi
}

BlacklistMenu() {
    echo "(I) It is highly recommended that add it into blacklist."
    echo -n "(Q) Do you want to add it into blacklist? (y/n) "

    while : ; do
        read yorn
        case $yorn in
            y) Add2Blacklist $1
               break;;
            n)
               break;;
            *) echo "(I) Please choose [y] or [n]"
               echo -n "(A) ";;
        esac
    done
}

Need2Blacklist() {
    ${checkmod} > ${modfile} 2> ${trash}

    grep -q "usbtouchscreen" ${modfile}
    if [ $? -eq 0 ]; then
        echo "(I) Found kernel module usbtouchscreen."
        BlacklistMenu "usbtouchscreen"
    else
        grep -q "touchkitusb" ${modfile}
        if [ $? -eq 0 ]; then
            echo "(I) Found kernel module touchkitusb."
            BlacklistMenu "touchkitusb"
        fi
    fi

    rm -f ${modfile}
}

Need2BuildModule() {
    ${checkmod} > ${modfile} 2> ${trash}

    grep -q "usbtouchscreen" ${modfile}
    if [ $? -eq 1 ]; then
        grep -q "touchkitusb" ${modfile}
        if [ $? -eq 1 ]; then
            echo "(W) No suitable kernel module found."
            echo "(I) The user must build tkusb kernel module for touch controller."
            echo "(I) For details, see the document \"How to build module.pdf\"."
        fi
    fi

    rm -f ${modfile}
}

CheckUSBType() {
    ${checkusb} > ${usbfile} 2> ${trash}

    grep -q "Human Interface Device" ${usbfile}
    if [ $? -eq 0 ]; then
        echo "(I) Found a HID compliant touch controller."
        Need2Blacklist
    else
        grep -q "Vendor Specific Protocol" ${usbfile}
        if [ $? -eq 0 ]; then
            echo "(I) Found a non-HID compliant touch controller."
            Need2BuildModule
        else
            echo "(W) Skip to check USB type."
        fi
    fi

    rm -f ${usbfile}
}

CheckInterface() {
    echo ""
    echo "(Q) Which interface controller do you use?"
    echo -n "(I) [1] RS232 [2] PS/2 [3] USB : "

    while : ; do
        read interface
        case $interface in
            1) echo "(Q) Which COM port will be connected? e.g. /dev/ttyS0 (COM1)"
               echo -n "(A) Please input: "
               read device # How to verify the input string???
               echo ""
               break;;
            2) echo "(I) Using interface: PS/2"
               echo "(I) Please make sure the kernel module for PS/2 controller is available."
               echo "(I) For details, see the document \"How to rebuild kernel.pdf\".\n"
               device="/dev/serio_raw0"
               break;;
            3) echo "(I) Using interface: USB"
               device="usbauto"
               CheckUSBType
               echo ""
               break;;
            *) echo "(I) Please choose [1], [2] or [3]"
               echo -n "(A) ";;
        esac
    done
}

AddConfiguration() {
    echo "(I) Add touch configuration into $1."
    sed -i '/ServerLayout/a\\tInputDevice\t"'${identifier}'" "SendCoreEvents"' $1

    filelines=`cat $1 | wc -l`
    sed -i ''${filelines}'a\### Touch Configuration Begin ###\
Section "InputDevice"\
	Identifier "'${identifier}'"\
	Driver "egalax"\
        Option "Device" "'$2'"\
        Option "Parameters" "'${paramfile}'"\
	Option "ScreenNo" "0"\
EndSection\
### Touch Configuration End ###' $1
}

RemoveConfiguration() {
    rm -f ${paramfile}

    sed -i '/"'${identifier}'" "SendCoreEvents"/d' $1
    sed -i '/### Touch Configuration Begin ###/,/### Touch Configuration End ###/d' $1
    echo "(I) Removed touch configuration in $1."
}

EditConfigFile() {
    if [ -w ${xfilepath} ]; then
        echo "(I) Found X configuration file xorg.conf in /etc/X11."
        grep -q "### Touch Configuration Begin ###" ${xfilepath}
        if [ $? -eq 1 ]; then
            AddConfiguration ${xfilepath} ${device}
        else
            RemoveConfiguration ${xfilepath}
            AddConfiguration ${xfilepath} ${device}
        fi
    elif [ -w ${xfilepath2} ]; then
        echo "(I) Found X configuration file XF86Config in /etc/X11."
        grep -q "### Touch Configuration Begin ###" ${xfilepath2}
        if [ $? -eq 1 ]; then
            AddConfiguration ${xfilepath2} ${device}
        else
            RemoveConfiguration ${xfilepath2}
            AddConfiguration ${xfilepath2} ${device}
        fi
    else
        echo "(E) No X configuration file found.\n"
        exit 1
    fi
}

RemoveDriver() {
    if [ -e ${installpath}/${driver} ]; then
        rm -f ${installpath}/${driver}
    fi

    temppath=`find ${installpath} -name ${utility}`
    driverpath=${temppath%/*}
    if [ -z ${driverpath} ]; then
        echo "(E) The driver directory has been removed already.\n"
        exit 1
    elif [ -d ${driverpath} ]; then
        echo "(I) Removed TouchKit driver from ${driverpath}."
        rm -rf ${driverpath}
    fi

    if [ -d ${shortcutpath} ]; then
        if [ -L ${shortcutpath}/${utility} ]; then
            echo "(I) Removed ${utility} utility shortcut."
            rm -f ${shortcutpath}/${utility}
        fi
    fi
}

RemoveXorgModule() {
    if [ -z ${mousepath} ]; then
        echo "(E) No X input modules directory found.\n"
        exit 1
    fi

    if [ ${mouse} = "mouse_drv.so" ]; then
        xmodule="egalax_drv.so"
    else
        xmodule="egalax_drv.o"
    fi

    if [ -e ${xpath}/${xmodule} ]; then
        echo "(I) Removed X module."
        rm -f ${xpath}/${xmodule}
    fi
}

RestoreConfigFile() {
    if [ -w ${xfilepath} ]; then
        echo "(I) Found X configuration file xorg.conf in /etc/X11."
        RemoveConfiguration ${xfilepath}
    elif [ -w ${xfilepath2} ]; then
        echo "(I) Found X configuration file XF86Config in /etc/X11."
        RemoveConfiguration ${xfilepath2}
    else
        echo "(E) No X configuration file found.\n"
        exit 1
    fi
}

RestoreBlacklist() {
    if [ -e ${blacklistpath} ]; then
        grep -q "blacklist usbtouchscreen" ${blacklistpath}
        if [ $? -eq 0 ]; then
            sed -i '/# add by eeti/,/blacklist usbtouchscreen/d' ${blacklistpath}
            echo "(I) Removed blacklist usbtouchscreen from ${blacklistpath}"
        else
            grep -q "blacklist touchkitusb" ${blacklistpath}
            if [ $? -eq 0 ]; then
                sed -i '/# add by eeti/,/blacklist touchkitusb/d' ${blacklistpath}
                echo "(I) Removed blacklist touchkitusb from ${blacklistpath}"
            fi
        fi
    fi
}

clear
ShowTitle

if [ $# = 0 ]; then
    echo "(I) Begin to setup TouchKit Linux driver."
    CheckPermission
    ExtractDriver
    CopyXorgMudule
    CheckInterface
    EditConfigFile
    echo ""
    echo "(I) Please reboot the system for some changes to take effect."
elif [ $# = 1 ]; then
    if [ $1 = "uninstall" ]; then
        echo "(I) Begin to remove TouchKit Linux driver."
        CheckPermission
        RemoveDriver
        RemoveXorgModule
        RestoreConfigFile
        RestoreBlacklist
        echo ""
        echo "(I) The TouchKit driver has been removed successfully."
        echo "(I) Please reboot the system for some changes to take effect."
    elif [ $1 = "version" ]; then
        echo "(I) Version: ${version}"
    else
        echo "(I) Usage: sh $0 or sh $0 uninstall"
    fi
else
    echo "(I) Usage: sh $0 or sh $0 uninstall"
fi

echo ""
